#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Script Educativo: Análisis de Cabeceras de Seguridad HTTP
Autor: Maicken | Vida Tech
Propósito: Demostrar web scraping ético para análisis de seguridad web

⚠️ IMPORTANTE: Este script es solo para fines educativos
"""

import requests
import json
from datetime import datetime
from bs4 import BeautifulSoup
import time
import sys
import os

# Habilitar colores ANSI en Windows
if sys.platform == 'win32':
    os.system('color')
    # Intentar habilitar el modo de terminal virtual en Windows 10+
    try:
        import ctypes
        kernel32 = ctypes.windll.kernel32
        kernel32.SetConsoleMode(kernel32.GetStdHandle(-11), 7)
    except:
        pass

# Colores para la consola (Windows compatible)
class Colors:
    HEADER = '\033[95m'
    BLUE = '\033[94m'
    CYAN = '\033[96m'
    GREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'

def print_banner():
    """Muestra el banner del programa"""
    banner = f"""
{Colors.CYAN}╔════════════════════════════════════════════════════════════╗
║                                                            ║
║    🔒 SECURITY HEADERS SCANNER - HERRAMIENTA EDUCATIVA 🔒  ║
║                                                            ║
║    👨‍💻 Maicken | Vida Tech                                  ║
║    📚 Para estudiantes de ciberseguridad                   ║
║    ⚠️  Solo uso educativo y ético                          ║
║                                                            ║
╚════════════════════════════════════════════════════════════╝{Colors.ENDC}
    """
    print(banner)

def analizar_cabeceras_seguridad(url):
    """
    Analiza las cabeceras de seguridad HTTP de un sitio web
    Identifica presencia/ausencia de cabeceras críticas de seguridad
    """
    print(f"\n{Colors.BLUE}[*] Analizando: {url}{Colors.ENDC}")
    
    try:
        headers = {
            'User-Agent': 'Mozilla/5.0 (Security Analysis Tool - Educational)'
        }
        
        # Hacer petición con timeout
        response = requests.get(url, headers=headers, timeout=10, allow_redirects=True, verify=True)
        
        print(f"{Colors.GREEN}[✓] Conexión exitosa - Status Code: {response.status_code}{Colors.ENDC}")
        
        return response.headers, response.status_code, response.url
    
    except requests.exceptions.SSLError:
        print(f"{Colors.FAIL}[!] Error SSL: Certificado inválido o problema de seguridad{Colors.ENDC}")
        return None, None, None
    except requests.exceptions.ConnectionError:
        print(f"{Colors.FAIL}[!] Error de conexión: Sitio no disponible{Colors.ENDC}")
        return None, None, None
    except requests.exceptions.Timeout:
        print(f"{Colors.FAIL}[!] Timeout: El servidor tardó demasiado en responder{Colors.ENDC}")
        return None, None, None
    except Exception as e:
        print(f"{Colors.FAIL}[!] Error: {str(e)}{Colors.ENDC}")
        return None, None, None

def evaluar_cabeceras(headers, url_final, url_original):
    """
    Evalúa las cabeceras de seguridad y genera un reporte detallado
    """
    print(f"\n{Colors.CYAN}{'='*60}{Colors.ENDC}")
    print(f"{Colors.BOLD}📋 ANÁLISIS DE CABECERAS DE SEGURIDAD HTTP{Colors.ENDC}")
    print(f"{Colors.CYAN}{'='*60}{Colors.ENDC}")
    
    if url_final != url_original:
        print(f"{Colors.WARNING}⚠️  Redireccionado a: {url_final}{Colors.ENDC}")
    
    # Cabeceras críticas de seguridad
    cabeceras_criticas = {
        'Strict-Transport-Security': {
            'nombre': 'HSTS (HTTP Strict Transport Security)',
            'descripcion': 'Fuerza conexiones HTTPS',
            'riesgo_ausencia': 'ALTO - Vulnerable a ataques downgrade'
        },
        'Content-Security-Policy': {
            'nombre': 'CSP (Content Security Policy)',
            'descripcion': 'Previene XSS y ataques de inyección',
            'riesgo_ausencia': 'ALTO - Vulnerable a XSS'
        },
        'X-Frame-Options': {
            'nombre': 'X-Frame-Options',
            'descripcion': 'Previene clickjacking',
            'riesgo_ausencia': 'MEDIO - Vulnerable a clickjacking'
        },
        'X-Content-Type-Options': {
            'nombre': 'X-Content-Type-Options',
            'descripcion': 'Previene MIME sniffing',
            'riesgo_ausencia': 'MEDIO - Vulnerable a MIME sniffing'
        },
        'Referrer-Policy': {
            'nombre': 'Referrer-Policy',
            'descripcion': 'Controla información de referencia',
            'riesgo_ausencia': 'BAJO - Fuga de información'
        },
        'Permissions-Policy': {
            'nombre': 'Permissions-Policy',
            'descripcion': 'Controla permisos de navegador',
            'riesgo_ausencia': 'BAJO - Control de permisos limitado'
        },
        'X-XSS-Protection': {
            'nombre': 'X-XSS-Protection',
            'descripcion': 'Protección XSS del navegador',
            'riesgo_ausencia': 'BAJO - Protección básica ausente'
        }
    }
    
    presentes = 0
    ausentes = 0
    
    print(f"\n{Colors.GREEN}✓ CABECERAS PRESENTES:{Colors.ENDC}")
    for header, info in cabeceras_criticas.items():
        if header in headers:
            presentes += 1
            valor = headers[header]
            # Truncar valor si es muy largo
            valor_mostrar = valor[:80] + '...' if len(valor) > 80 else valor
            print(f"\n   🛡️  {info['nombre']}")
            print(f"      └─ Valor: {valor_mostrar}")
            print(f"      └─ Función: {info['descripcion']}")
    
    if presentes == 0:
        print(f"   {Colors.WARNING}⚠️  Ninguna cabecera de seguridad crítica encontrada{Colors.ENDC}")
    
    print(f"\n{Colors.FAIL}✗ CABECERAS AUSENTES (VULNERABILIDADES):{Colors.ENDC}")
    for header, info in cabeceras_criticas.items():
        if header not in headers:
            ausentes += 1
            print(f"\n   ⚠️  {info['nombre']}")
            print(f"      └─ Riesgo: {info['riesgo_ausencia']}")
            print(f"      └─ Función: {info['descripcion']}")
    
    # Otras cabeceras interesantes
    print(f"\n{Colors.BLUE}ℹ️  OTRAS CABECERAS DE INTERÉS:{Colors.ENDC}")
    otras_cabeceras = ['Server', 'X-Powered-By', 'Set-Cookie', 'Access-Control-Allow-Origin']
    
    for header in otras_cabeceras:
        if header in headers:
            valor = headers[header]
            valor_mostrar = valor[:60] + '...' if len(valor) > 60 else valor
            print(f"   • {header}: {valor_mostrar}")
    
    # Puntuación de seguridad
    total_cabeceras = len(cabeceras_criticas)
    puntuacion = (presentes / total_cabeceras) * 100
    
    print(f"\n{Colors.BOLD}{'='*60}{Colors.ENDC}")
    print(f"{Colors.BOLD}🎯 PUNTUACIÓN DE SEGURIDAD: {puntuacion:.1f}/100{Colors.ENDC}")
    print(f"   • Cabeceras presentes: {presentes}/{total_cabeceras}")
    print(f"   • Cabeceras ausentes: {ausentes}/{total_cabeceras}")
    
    if puntuacion >= 80:
        nivel = f"{Colors.GREEN}EXCELENTE ✓{Colors.ENDC}"
    elif puntuacion >= 60:
        nivel = f"{Colors.BLUE}BUENO{Colors.ENDC}"
    elif puntuacion >= 40:
        nivel = f"{Colors.WARNING}REGULAR ⚠️{Colors.ENDC}"
    else:
        nivel = f"{Colors.FAIL}DEFICIENTE ✗{Colors.ENDC}"
    
    print(f"   • Nivel de seguridad: {nivel}")
    print(f"{Colors.BOLD}{'='*60}{Colors.ENDC}")

def agregar_sitios_personalizados():
    """
    Permite al usuario agregar múltiples sitios para analizar
    """
    sitios = []
    
    print(f"\n{Colors.CYAN}{'='*60}{Colors.ENDC}")
    print(f"{Colors.BOLD}📝 AGREGAR SITIOS WEB PARA ANÁLISIS{Colors.ENDC}")
    print(f"{Colors.CYAN}{'='*60}{Colors.ENDC}")
    print(f"\n{Colors.WARNING}Instrucciones:{Colors.ENDC}")
    print(f"  • Ingresa las URLs completas (ej: https://ejemplo.com)")
    print(f"  • Presiona ENTER sin escribir nada para finalizar")
    print(f"  • O escribe 'ejemplos' para usar sitios de demostración\n")
    
    contador = 1
    while True:
        try:
            sitio = input(f"{Colors.GREEN}[{contador}] URL del sitio: {Colors.ENDC}").strip()
            
            if sitio.lower() == 'ejemplos':
                sitios = [
                    'https://www.google.com',
                    'https://www.facebook.com',
                    'https://github.com',
                    'http://neverssl.com',
                    'https://www.youtube.com'
                ]
                print(f"{Colors.BLUE}\n[*] Cargados 5 sitios de ejemplo{Colors.ENDC}")
                break
            
            if not sitio:
                if len(sitios) == 0:
                    print(f"{Colors.WARNING}⚠️  Debes agregar al menos un sitio{Colors.ENDC}")
                    continue
                break
            
            # Validar formato básico
            if not sitio.startswith(('http://', 'https://')):
                print(f"{Colors.FAIL}⚠️  URL debe comenzar con http:// o https://{Colors.ENDC}")
                continue
            
            sitios.append(sitio)
            contador += 1
            print(f"{Colors.GREEN}   ✓ Agregado{Colors.ENDC}")
            
        except KeyboardInterrupt:
            print(f"\n{Colors.WARNING}[!] Cancelado por el usuario{Colors.ENDC}")
            break
    
    return sitios

def generar_recomendaciones(total_analisis, sitios_buenos, sitios_regulares, sitios_malos):
    """
    Genera recomendaciones basadas en los análisis realizados
    """
    print(f"\n{Colors.GREEN}{'='*60}{Colors.ENDC}")
    print(f"{Colors.BOLD}{Colors.GREEN}🛡️  RECOMENDACIONES DE SEGURIDAD:{Colors.ENDC}")
    
    print(f"\n{Colors.CYAN}📊 Resumen del Análisis:{Colors.ENDC}")
    print(f"   • Total de sitios analizados: {total_analisis}")
    print(f"   • {Colors.GREEN}Nivel EXCELENTE/BUENO: {sitios_buenos}{Colors.ENDC}")
    print(f"   • {Colors.WARNING}Nivel REGULAR: {sitios_regulares}{Colors.ENDC}")
    print(f"   • {Colors.FAIL}Nivel DEFICIENTE: {sitios_malos}{Colors.ENDC}")
    
    print(f"\n{Colors.BOLD}💡 Buenas Prácticas para Implementar:{Colors.ENDC}")
    print(f"""
    1. ✓ HSTS: Implementar para forzar HTTPS siempre
    2. ✓ CSP: Definir política de contenido estricta
    3. ✓ X-Frame-Options: DENY o SAMEORIGIN para prevenir clickjacking
    4. ✓ X-Content-Type-Options: nosniff para prevenir MIME sniffing
    5. ✓ Referrer-Policy: Controlar información compartida
    6. ✓ HTTPS: Usar siempre certificados SSL/TLS válidos
    7. ✓ Actualizar regularmente configuraciones de seguridad
    8. ✓ Monitorear cabeceras con herramientas automatizadas
    """)
    
    print(f"\n{Colors.BOLD}🔗 Recursos Educativos:{Colors.ENDC}")
    print(f"   • OWASP Secure Headers Project")
    print(f"   • Mozilla Observatory")
    print(f"   • SecurityHeaders.com")
    print(f"   • Content Security Policy Reference")
    
    print(f"\n{Colors.BOLD}👨‍💻 Maicken | Vida Tech{Colors.ENDC}")
    print(f"   📚 Contenido educativo de ciberseguridad")
    print(f"{Colors.GREEN}{'='*60}{Colors.ENDC}")

def main():
    """Función principal"""
    print_banner()
    
    print(f"\n{Colors.BOLD}🎓 CLASE: Web Scraping de Cabeceras de Seguridad HTTP{Colors.ENDC}")
    print(f"{Colors.CYAN}Fecha: {datetime.now().strftime('%d/%m/%Y %H:%M:%S')}{Colors.ENDC}")
    print(f"{Colors.CYAN}Instructor: Maicken | Vida Tech{Colors.ENDC}")
    
    # Obtener sitios del usuario
    sitios = agregar_sitios_personalizados()
    
    if not sitios:
        print(f"\n{Colors.FAIL}[!] No se agregaron sitios para analizar. Finalizando...{Colors.ENDC}")
        return
    
    print(f"\n{Colors.GREEN}[✓] Se analizarán {len(sitios)} sitio(s){Colors.ENDC}")
    time.sleep(1)
    
    # Contadores para estadísticas
    sitios_buenos = 0
    sitios_regulares = 0
    sitios_malos = 0
    total_analisis = 0
    
    # Analizar cada sitio
    for idx, sitio in enumerate(sitios, 1):
        print(f"\n\n{Colors.HEADER}{'='*60}{Colors.ENDC}")
        print(f"{Colors.BOLD}🌐 ANÁLISIS [{idx}/{len(sitios)}]: {sitio}{Colors.ENDC}")
        print(f"{Colors.HEADER}{'='*60}{Colors.ENDC}")
        
        headers, status_code, url_final = analizar_cabeceras_seguridad(sitio)
        
        if headers:
            evaluar_cabeceras(headers, url_final, sitio)
            total_analisis += 1
            
            # Calcular puntuación para estadísticas
            cabeceras_criticas = ['Strict-Transport-Security', 'Content-Security-Policy', 
                                'X-Frame-Options', 'X-Content-Type-Options', 
                                'Referrer-Policy', 'Permissions-Policy', 'X-XSS-Protection']
            presentes = sum(1 for h in cabeceras_criticas if h in headers)
            puntuacion = (presentes / len(cabeceras_criticas)) * 100
            
            if puntuacion >= 60:
                sitios_buenos += 1
            elif puntuacion >= 40:
                sitios_regulares += 1
            else:
                sitios_malos += 1
        else:
            print(f"\n{Colors.FAIL}[✗] No se pudo analizar el sitio{Colors.ENDC}")
        
        # Pausa entre análisis
        if idx < len(sitios):
            time.sleep(1)
    
    # Generar recomendaciones finales
    generar_recomendaciones(total_analisis, sitios_buenos, sitios_regulares, sitios_malos)
    
    # Nota educativa final
    print(f"\n{Colors.WARNING}⚠️  NOTA EDUCATIVA:{Colors.ENDC}")
    print(f"""
    Este script demuestra técnicas de web scraping ético:
    
    • Analiza cabeceras HTTP públicas de forma legal
    • Identifica vulnerabilidades de configuración
    • Educa sobre mejores prácticas de seguridad
    • Respeta protocolos estándar de la web
    
    {Colors.BOLD}💡 Para tus alumnos:{Colors.ENDC}
    - Las cabeceras HTTP son información pública
    - El análisis de seguridad debe ser ético y autorizado
    - Estas herramientas ayudan a MEJORAR la seguridad
    - Siempre obtén permiso antes de analizar sitios corporativos
    
    {Colors.BOLD}👨‍💻 Sígueme en TikTok: Maicken | Vida Tech{Colors.ENDC}
    """)
    
    print(f"\n{Colors.GREEN}[✓] Script finalizado. ¡Gracias por aprender ciberseguridad! 🎓{Colors.ENDC}\n")

if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        print(f"\n\n{Colors.WARNING}[!] Script interrumpido por el usuario{Colors.ENDC}")
    except Exception as e:
        print(f"\n{Colors.FAIL}[!] Error inesperado: {str(e)}{Colors.ENDC}")
